! MIT License
! 
! Copyright (c) 2018 Florian Goth
!
! Permission is hereby granted, free of charge, to any person obtaining a copy
! of this software and associated documentation files (the "Software"), to deal
! in the Software without restriction, including without limitation the rights 
! to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
! copies of the Software, and to permit persons to whom the Software is
! furnished to do so, subject to the following conditions:
! 
! The above copyright notice and this permission notice shall be included in 
! all copies or substantial portions of the Software.
! 
! THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
! OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
! FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
! THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
! LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
! FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
! DEALINGS IN THE SOFTWARE.


!--------------------------------------------------------------------
!> @author
!> Florian Goth
!
!> @brief 
!> This function takes an h5 file and reads the matrix
!
!> @param[in] filename The filename of the HDF5 file
!> @result The matrix
!--------------------------------------------------------------------    
    function h5readmat(filename) result(A)
    use hdf5
    implicit none
    character(LEN=*), intent(in) :: filename
    real(kind=kind(0.D0)), allocatable, dimension(:, :) :: At
    complex(kind=kind(0.D0)), allocatable, dimension(:, :) :: A
    character(LEN=8), parameter :: dsetname = "Dataset1"     ! Dataset name from Mathematica
    integer :: ndim, i, j, tmpint
    integer :: error, rank
    integer(HSIZE_T), allocatable, dimension(:) :: dims, maxdims
    integer(hid_t) :: file_id, dset_id, dspace_id

    ! init HDF5 and open file
    call h5open_f(error)
    call h5fopen_f(filename, H5F_ACC_RDONLY_F, file_id, error)
    ! open dataset and dataspace
    call h5dopen_f(file_id, dsetname, dset_id, error)
    call h5dget_space_f(dset_id, dspace_id, error)
    ! Get the rank of the object
    call h5sget_simple_extent_ndims_f(dspace_id, rank, error)
    allocate(dims(rank), maxdims(rank))
    ! Get the actual matrix dimensions
    call h5sget_simple_extent_dims_f(dspace_id, dims, maxdims, error)
    allocate(At(dims(1), dims(2)), A(dims(1), dims(2)))
    ! read the actual data
    call h5dread_f(dset_id, H5T_IEEE_F64LE, At, dims, error)
    ! tidy up and close the file.
    call h5sclose_f(dspace_id, error)
    call h5dclose_f(dset_id, error)
    call h5fclose_f(file_id, error)
    call h5close_f(error)
    do i = 1, dims(1)
        do j = 1, dims(2)
            A(i,j) = At(i,j)
        enddo
    enddo
    end function

!--------------------------------------------------------------------
!> @author
!> Florian Goth
!
!> @brief 
!> This function takes an nc(NetCDF) file and reads the matrix
!
!> @param[in] filename The filename of the NetCDF file
!> @result The matrix
!--------------------------------------------------------------------    
    function ncreadmat(filename) result(A)
    use netcdf
    implicit none
    character(LEN=*), intent(in) :: filename
    real(kind=kind(0.D0)), allocatable, dimension(:, :) :: At
    complex(kind=kind(0.D0)), allocatable, dimension(:, :) :: A
    character(LEN=8), parameter :: dsetname = "Dataset1"     ! Dataset name from Mathematica
    integer :: i, j
    integer :: rank
    integer, allocatable, dimension(:) :: dims, mydimids
    integer :: mystat
    integer :: ncid, varid

    ! open the file
    mystat = nf90_open(filename, NF90_NOWRITE, ncid)
    ! obtain an ID for the variable/dataset that we require
    mystat = nf90_inq_varid(ncid, "Dataset1", varid)
    ! query rank of object 
    mystat = nf90_inquire_variable(ncid, varid, ndims=rank)
    allocate(dims(rank), mydimids(rank))
    mystat = nf90_inquire_variable(ncid, varid, dimids=mydimids)
    ! map the dimension ids to the actual values
    do i = 1, rank
       mystat = nf90_inquire_dimension(ncid, mydimids(i), len=dims(i))
    enddo
    ! Read the matrix
    allocate(At(dims(1), dims(2)), A(dims(1), dims(2)))
    mystat = nf90_get_var(ncid, varid, At)

    ! close the file
    mystat = nf90_close(ncid)
    do i = 1, dims(1)
        do j = 1, dims(2)
            A(i,j) = At(i,j)
        enddo
    enddo
    deallocate(dims)
    end function
